<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Require admin role
$auth->requireAdmin();

$page_title = 'Pengguna';
$current_user = $auth->getCurrentUser();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create_user':
                $name = sanitize($_POST['name']);
                $nip = sanitize($_POST['nip']);
                $email = sanitize($_POST['email']);
                $position = sanitize($_POST['position']);
                $role = sanitize($_POST['role']);
                $password = $_POST['password'];
                
                if (strlen($password) < 6) {
                    setFlashMessage('error', 'Password minimal 6 karakter!');
                } else {
                    $result = $auth->register($name, $nip, $email, $position, $password, $role);
                    
                    if ($result['success']) {
                        setFlashMessage('success', 'Pengguna berhasil dibuat!');
                    } else {
                        setFlashMessage('error', $result['message']);
                    }
                }
                break;
                
            case 'update_user':
                $user_id = intval($_POST['user_id']);
                $name = sanitize($_POST['name']);
                $nip = sanitize($_POST['nip']);
                $email = sanitize($_POST['email']);
                $position = sanitize($_POST['position']);
                $role = sanitize($_POST['role']);
                $is_active = isset($_POST['is_active']) ? 1 : 0;
                
                try {
                    $stmt = $pdo->prepare("UPDATE users SET name = ?, nip = ?, email = ?, position = ?, role = ?, is_active = ?, updated_at = ? WHERE id = ?");
                    $stmt->execute([$name, $nip, $email, $position, $role, $is_active, date('Y-m-d H:i:s'), $user_id]);
                    
                    setFlashMessage('success', 'Data pengguna berhasil diperbarui!');
                } catch (PDOException $e) {
                    setFlashMessage('error', 'Gagal memperbarui pengguna: ' . $e->getMessage());
                }
                break;
                
            case 'reset_password':
                $user_id = intval($_POST['user_id']);
                $new_password = $_POST['new_password'];
                
                if (strlen($new_password) < 6) {
                    setFlashMessage('error', 'Password minimal 6 karakter!');
                } else {
                    try {
                        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                        $stmt = $pdo->prepare("UPDATE users SET password = ?, updated_at = ? WHERE id = ?");
                        $stmt->execute([$hashed_password, date('Y-m-d H:i:s'), $user_id]);
                        
                        setFlashMessage('success', 'Password berhasil direset!');
                    } catch (PDOException $e) {
                        setFlashMessage('error', 'Gagal mereset password: ' . $e->getMessage());
                    }
                }
                break;
                
            case 'delete_user':
                $user_id = intval($_POST['user_id']);
                
                if ($user_id === $current_user['id']) {
                    setFlashMessage('error', 'Tidak dapat menghapus akun Anda sendiri!');
                } else {
                    try {
                        // Check if user has permissions
                        $stmt = $pdo->prepare("SELECT COUNT(*) FROM permissions WHERE user_id = ?");
                        $stmt->execute([$user_id]);
                        $permission_count = $stmt->fetchColumn();
                        
                        if ($permission_count > 0) {
                            setFlashMessage('error', 'Tidak dapat menghapus pengguna yang memiliki riwayat izin!');
                        } else {
                            $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                            $stmt->execute([$user_id]);
                            
                            setFlashMessage('success', 'Pengguna berhasil dihapus!');
                        }
                    } catch (PDOException $e) {
                        setFlashMessage('error', 'Gagal menghapus pengguna: ' . $e->getMessage());
                    }
                }
                break;
        }
    }
}

// Get users with pagination
$page = intval($_GET['page'] ?? 1);
$limit = 10;
$offset = ($page - 1) * $limit;
$search = sanitize($_GET['search'] ?? '');
$role_filter = sanitize($_GET['role'] ?? '');

try {
    // Build where clause
    $where_conditions = [];
    $params = [];
    
    if ($search) {
        $where_conditions[] = "(name LIKE ? OR nip LIKE ? OR email LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }
    
    if ($role_filter) {
        $where_conditions[] = "role = ?";
        $params[] = $role_filter;
    }
    
    $where_clause = $where_conditions ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    // Get total count
    $count_sql = "SELECT COUNT(*) FROM users $where_clause";
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($params);
    $total_users = $stmt->fetchColumn();
    
    // Get users
    $sql = "SELECT * FROM users $where_clause ORDER BY created_at DESC LIMIT $limit OFFSET $offset";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $users = $stmt->fetchAll();
    
    $total_pages = ceil($total_users / $limit);
} catch (PDOException $e) {
    $users = [];
    $total_users = 0;
    $total_pages = 0;
}

// Get user statistics
try {
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN role = 'admin' THEN 1 ELSE 0 END) as admin_count,
        SUM(CASE WHEN role = 'user' THEN 1 ELSE 0 END) as user_count,
        SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_count
        FROM users");
    $stmt->execute();
    $user_stats = $stmt->fetch();
} catch (PDOException $e) {
    $user_stats = [
        'total' => 0,
        'admin_count' => 0,
        'user_count' => 0,
        'active_count' => 0
    ];
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <?php 
                generateBreadcrumb([
                    ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                    ['title' => 'Pengguna', 'url' => '']
                ]); 
                ?>
                
                <!-- Flash Messages -->
                <?php showFlashMessage(); ?>
                
                <div class="mb-6 flex justify-between items-center">
                    <div>
                        <h1 class="text-2xl font-bold text-gray-800">Manajemen Pengguna</h1>
                        <p class="text-gray-600 mt-1">Kelola pengguna sistem</p>
                    </div>
                    <button onclick="openCreateModal()" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center">
                        <i data-feather="plus" class="w-4 h-4 mr-2"></i>
                        Tambah Pengguna
                    </button>
                </div>

                <!-- Statistics Cards -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Total Pengguna</p>
                                <h3 class="text-2xl font-bold mt-1"><?php echo $user_stats['total']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                                <i data-feather="users" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Administrator</p>
                                <h3 class="text-2xl font-bold mt-1 text-purple-600"><?php echo $user_stats['admin_count']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                                <i data-feather="shield" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Pegawai</p>
                                <h3 class="text-2xl font-bold mt-1 text-green-600"><?php echo $user_stats['user_count']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-green-100 text-green-600">
                                <i data-feather="user" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Aktif</p>
                                <h3 class="text-2xl font-bold mt-1 text-blue-600"><?php echo $user_stats['active_count']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                                <i data-feather="user-check" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="bg-white rounded-lg shadow p-6 mb-6">
                    <form method="GET" class="flex flex-wrap gap-4 items-end">
                        <div class="flex-1 min-w-64">
                            <label class="block text-sm font-medium text-gray-700 mb-1">Pencarian</label>
                            <input type="text" name="search" value="<?php echo escape($search); ?>" 
                                   placeholder="Nama, NIP, atau Email..." 
                                   class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Role</label>
                            <select name="role" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                                <option value="">Semua Role</option>
                                <option value="admin" <?php echo ($role_filter === 'admin') ? 'selected' : ''; ?>>Administrator</option>
                                <option value="user" <?php echo ($role_filter === 'user') ? 'selected' : ''; ?>>Pegawai</option>
                            </select>
                        </div>
                        <div>
                            <button type="submit" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center">
                                <i data-feather="search" class="w-4 h-4 mr-2"></i>
                                Cari
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Users Table -->
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-semibold">Daftar Pengguna</h2>
                    </div>
                    <div class="overflow-x-auto">
                        <?php if (!empty($users)): ?>
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Pengguna</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Kontak</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Role</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Terdaftar</th>
                                    <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Aksi</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($users as $user): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                                                <span class="text-blue-600 font-medium text-sm">
                                                    <?php echo strtoupper(substr($user['name'], 0, 2)); ?>
                                                </span>
                                            </div>
                                            <div class="ml-4">
                                                <div class="text-sm font-medium text-gray-900"><?php echo escape($user['name']); ?></div>
                                                <div class="text-sm text-gray-500">NIP: <?php echo escape($user['nip']); ?></div>
                                                <div class="text-xs text-gray-400"><?php echo escape($user['position']); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm text-gray-900"><?php echo escape($user['email']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php if ($user['role'] === 'admin'): ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                            <i data-feather="shield" class="w-3 h-3 mr-1"></i>
                                            Administrator
                                        </span>
                                        <?php else: ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                            <i data-feather="user" class="w-3 h-3 mr-1"></i>
                                            Pegawai
                                        </span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php if ($user['is_active']): ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                            <span class="w-1.5 h-1.5 bg-green-400 rounded-full mr-1.5"></span>
                                            Aktif
                                        </span>
                                        <?php else: ?>
                                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                            <span class="w-1.5 h-1.5 bg-red-400 rounded-full mr-1.5"></span>
                                            Nonaktif
                                        </span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo formatTanggalIndonesia($user['created_at'], 'd/m/Y'); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                        <div class="flex justify-end space-x-2">
                                            <button onclick="openEditModal(<?php echo htmlspecialchars(json_encode($user)); ?>)" 
                                                    class="text-blue-600 hover:text-blue-900">
                                                <i data-feather="edit" class="w-4 h-4"></i>
                                            </button>
                                            <button onclick="openResetPasswordModal(<?php echo $user['id']; ?>, '<?php echo escape($user['name']); ?>')" 
                                                    class="text-yellow-600 hover:text-yellow-900">
                                                <i data-feather="key" class="w-4 h-4"></i>
                                            </button>
                                            <?php if ($user['id'] !== $current_user['id']): ?>
                                            <button onclick="deleteUser(<?php echo $user['id']; ?>, '<?php echo escape($user['name']); ?>')" 
                                                    class="text-red-600 hover:text-red-900">
                                                <i data-feather="trash-2" class="w-4 h-4"></i>
                                            </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        
                        <!-- Pagination -->
                        <?php if ($total_pages > 1): ?>
                        <div class="px-6 py-4 border-t border-gray-200">
                            <?php echo generatePagination($page, $total_pages, $_GET); ?>
                        </div>
                        <?php endif; ?>
                        
                        <?php else: ?>
                        <div class="p-12 text-center">
                            <i data-feather="users" class="w-16 h-16 mx-auto text-gray-300 mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">Tidak ada pengguna</h3>
                            <p class="text-gray-500">Belum ada pengguna yang terdaftar.</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Create User Modal -->
    <div id="createModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Tambah Pengguna Baru</h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="create_user">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Nama Lengkap</label>
                        <input type="text" name="name" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">NIP</label>
                        <input type="text" name="nip" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Email</label>
                        <input type="email" name="email" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Jabatan</label>
                        <input type="text" name="position" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Role</label>
                        <select name="role" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                            <option value="user">Pegawai</option>
                            <option value="admin">Administrator</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Password</label>
                        <input type="password" name="password" required minlength="6" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div class="flex justify-end space-x-2">
                        <button type="button" onclick="closeModal('createModal')" class="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50">
                            Batal
                        </button>
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                            Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit User Modal -->
    <div id="editModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Edit Pengguna</h3>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="update_user">
                    <input type="hidden" name="user_id" id="edit_user_id">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Nama Lengkap</label>
                        <input type="text" name="name" id="edit_name" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">NIP</label>
                        <input type="text" name="nip" id="edit_nip" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Email</label>
                        <input type="email" name="email" id="edit_email" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Jabatan</label>
                        <input type="text" name="position" id="edit_position" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Role</label>
                        <select name="role" id="edit_role" required class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                            <option value="user">Pegawai</option>
                            <option value="admin">Administrator</option>
                        </select>
                    </div>
                    <div class="flex items-center">
                        <input type="checkbox" name="is_active" id="edit_is_active" class="mr-2">
                        <label for="edit_is_active" class="text-sm font-medium text-gray-700">Aktif</label>
                    </div>
                    <div class="flex justify-end space-x-2">
                        <button type="button" onclick="closeModal('editModal')" class="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50">
                            Batal
                        </button>
                        <button type="submit" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                            Update
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Reset Password Modal -->
    <div id="resetPasswordModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Reset Password</h3>
                <p class="text-sm text-gray-600 mb-4">Reset password untuk: <span id="reset_user_name" class="font-medium"></span></p>
                <form method="POST" class="space-y-4">
                    <input type="hidden" name="action" value="reset_password">
                    <input type="hidden" name="user_id" id="reset_user_id">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Password Baru</label>
                        <input type="password" name="new_password" required minlength="6" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                        <p class="text-xs text-gray-500 mt-1">Minimal 6 karakter</p>
                    </div>
                    <div class="flex justify-end space-x-2">
                        <button type="button" onclick="closeModal('resetPasswordModal')" class="px-4 py-2 text-gray-600 border border-gray-300 rounded-lg hover:bg-gray-50">
                            Batal
                        </button>
                        <button type="submit" class="px-4 py-2 bg-yellow-600 text-white rounded-lg hover:bg-yellow-700">
                            Reset
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        feather.replace();
        
        function openCreateModal() {
            document.getElementById('createModal').classList.remove('hidden');
        }
        
        function openEditModal(user) {
            document.getElementById('edit_user_id').value = user.id;
            document.getElementById('edit_name').value = user.name;
            document.getElementById('edit_nip').value = user.nip;
            document.getElementById('edit_email').value = user.email;
            document.getElementById('edit_position').value = user.position;
            document.getElementById('edit_role').value = user.role;
            document.getElementById('edit_is_active').checked = user.is_active == 1;
            
            document.getElementById('editModal').classList.remove('hidden');
        }
        
        function openResetPasswordModal(userId, userName) {
            document.getElementById('reset_user_id').value = userId;
            document.getElementById('reset_user_name').textContent = userName;
            
            document.getElementById('resetPasswordModal').classList.remove('hidden');
        }
        
        function closeModal(modalId) {
            document.getElementById(modalId).classList.add('hidden');
        }
        
        function deleteUser(userId, userName) {
            if (confirm(`Yakin ingin menghapus pengguna "${userName}"?`)) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="action" value="delete_user">
                    <input type="hidden" name="user_id" value="${userId}">
                `;
                
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modals = ['createModal', 'editModal', 'resetPasswordModal'];
            modals.forEach(modalId => {
                const modal = document.getElementById(modalId);
                if (event.target === modal) {
                    closeModal(modalId);
                }
            });
        }
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>