<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Require admin role
$auth->requireAdmin();

$page_title = 'Pengaturan';
$current_user = $auth->getCurrentUser();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_app_settings':
                // Update app settings
                try {
                    $app_name = sanitize($_POST['app_name']);
                    $office_name = sanitize($_POST['office_name']);
                    $address = sanitize($_POST['address']);
                    $phone = sanitize($_POST['phone']);
                    $email = sanitize($_POST['email']);
                    
                    $stmt = $pdo->prepare("UPDATE settings SET value = ? WHERE name = ?");
                    $stmt->execute([$app_name, 'app_name']);
                    $stmt->execute([$office_name, 'office_name']);
                    $stmt->execute([$address, 'address']);
                    $stmt->execute([$phone, 'phone']);
                    $stmt->execute([$email, 'email']);
                    
                    setFlashMessage('success', 'Pengaturan aplikasi berhasil diperbarui!');
                } catch (PDOException $e) {
                    setFlashMessage('error', 'Gagal memperbarui pengaturan: ' . $e->getMessage());
                }
                break;
                
            case 'update_permission_types':
                // Update permission types
                try {
                    $types = json_decode($_POST['types'], true);
                    
                    // Clear existing types
                    $stmt = $pdo->prepare("DELETE FROM permission_types");
                    $stmt->execute();
                    
                    // Insert new types
                    $stmt = $pdo->prepare("INSERT INTO permission_types (name, description, max_duration_hours, created_at) VALUES (?, ?, ?, ?)");
                    
                    foreach ($types as $type) {
                        $stmt->execute([
                            sanitize($type['name']),
                            sanitize($type['description']),
                            intval($type['max_duration_hours']),
                            date('Y-m-d H:i:s')
                        ]);
                    }
                    
                    setFlashMessage('success', 'Jenis izin berhasil diperbarui!');
                } catch (PDOException $e) {
                    setFlashMessage('error', 'Gagal memperbarui jenis izin: ' . $e->getMessage());
                }
                break;
                
            case 'backup_database':
                // Create database backup
                try {
                    $backup_dir = dirname(__DIR__) . '/backups';
                    if (!is_dir($backup_dir)) {
                        mkdir($backup_dir, 0755, true);
                    }
                    
                    $backup_file = $backup_dir . '/siskim_backup_' . date('Y-m-d_H-i-s') . '.db';
                    
                    if (copy($database_path, $backup_file)) {
                        setFlashMessage('success', 'Backup database berhasil dibuat: ' . basename($backup_file));
                    } else {
                        setFlashMessage('error', 'Gagal membuat backup database');
                    }
                } catch (Exception $e) {
                    setFlashMessage('error', 'Gagal membuat backup: ' . $e->getMessage());
                }
                break;
        }
    }
}

// Get current settings
try {
    $stmt = $pdo->prepare("SELECT name, value FROM settings");
    $stmt->execute();
    $settings_raw = $stmt->fetchAll();
    
    $settings = [];
    foreach ($settings_raw as $setting) {
        $settings[$setting['name']] = $setting['value'];
    }
} catch (PDOException $e) {
    $settings = [];
}

// Get permission types
try {
    $stmt = $pdo->prepare("SELECT * FROM permission_types ORDER BY name");
    $stmt->execute();
    $permission_types = $stmt->fetchAll();
} catch (PDOException $e) {
    $permission_types = [];
}

// Get system stats
try {
    $stats = [];
    
    // Database size
    $stats['db_size'] = file_exists($database_path) ? formatFileSize(filesize($database_path)) : 'Unknown';
    
    // Total users
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM users");
    $stmt->execute();
    $stats['total_users'] = $stmt->fetchColumn();
    
    // Total permissions
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM permissions");
    $stmt->execute();
    $stats['total_permissions'] = $stmt->fetchColumn();
    
    // Recent activity (last 7 days)
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM permissions WHERE created_at >= datetime('now', '-7 days')");
    $stmt->execute();
    $stats['recent_activity'] = $stmt->fetchColumn();
    
} catch (PDOException $e) {
    $stats = [
        'db_size' => 'Error',
        'total_users' => 0,
        'total_permissions' => 0,
        'recent_activity' => 0
    ];
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <?php 
                generateBreadcrumb([
                    ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                    ['title' => 'Pengaturan', 'url' => '']
                ]); 
                ?>
                
                <!-- Flash Messages -->
                <?php showFlashMessage(); ?>
                
                <div class="mb-6">
                    <h1 class="text-2xl font-bold text-gray-800">Pengaturan Sistem</h1>
                    <p class="text-gray-600 mt-1">Kelola pengaturan aplikasi dan konfigurasi sistem</p>
                </div>

                <!-- Settings Tabs -->
                <div class="mb-6">
                    <nav class="flex space-x-8" aria-label="Tabs">
                        <button class="tab-button active border-blue-500 text-blue-600 whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm" 
                                data-tab="app" onclick="switchTab(this, 'app')">
                            Aplikasi
                        </button>
                        <button class="tab-button border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm"
                                data-tab="types" onclick="switchTab(this, 'types')">
                            Jenis Izin
                        </button>
                        <button class="tab-button border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 whitespace-nowrap py-2 px-1 border-b-2 font-medium text-sm"
                                data-tab="system" onclick="switchTab(this, 'system')">
                            Sistem
                        </button>
                    </nav>
                </div>

                <!-- App Settings Tab -->
                <div id="tab-app" class="tab-content">
                    <div class="bg-white rounded-lg shadow">
                        <div class="px-6 py-4 border-b border-gray-200">
                            <h2 class="text-lg font-semibold">Pengaturan Aplikasi</h2>
                            <p class="text-sm text-gray-600 mt-1">Konfigurasi informasi dasar aplikasi</p>
                        </div>
                        <form method="POST" class="p-6">
                            <input type="hidden" name="action" value="update_app_settings">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Nama Aplikasi</label>
                                    <input type="text" name="app_name" value="<?php echo escape($settings['app_name'] ?? 'SISKIM PN Watampone'); ?>" 
                                           class="block w-full border border-gray-300 rounded-lg py-2 px-3 focus:ring-blue-500 focus:border-blue-500" required>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Nama Instansi</label>
                                    <input type="text" name="office_name" value="<?php echo escape($settings['office_name'] ?? 'Pengadilan Negeri Watampone'); ?>" 
                                           class="block w-full border border-gray-300 rounded-lg py-2 px-3 focus:ring-blue-500 focus:border-blue-500" required>
                                </div>
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Alamat</label>
                                    <textarea name="address" rows="3" class="block w-full border border-gray-300 rounded-lg py-2 px-3 focus:ring-blue-500 focus:border-blue-500"><?php echo escape($settings['address'] ?? ''); ?></textarea>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Telepon</label>
                                    <input type="tel" name="phone" value="<?php echo escape($settings['phone'] ?? ''); ?>" 
                                           class="block w-full border border-gray-300 rounded-lg py-2 px-3 focus:ring-blue-500 focus:border-blue-500">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                                    <input type="email" name="email" value="<?php echo escape($settings['email'] ?? ''); ?>" 
                                           class="block w-full border border-gray-300 rounded-lg py-2 px-3 focus:ring-blue-500 focus:border-blue-500">
                                </div>
                            </div>
                            <div class="mt-6 flex justify-end">
                                <button type="submit" class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700">
                                    Simpan Perubahan
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Permission Types Tab -->
                <div id="tab-types" class="tab-content hidden">
                    <div class="bg-white rounded-lg shadow">
                        <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                            <div>
                                <h2 class="text-lg font-semibold">Jenis Izin</h2>
                                <p class="text-sm text-gray-600 mt-1">Kelola jenis-jenis izin yang tersedia</p>
                            </div>
                            <button onclick="addPermissionType()" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 flex items-center">
                                <i data-feather="plus" class="w-4 h-4 mr-2"></i>
                                Tambah Jenis
                            </button>
                        </div>
                        <div class="p-6">
                            <div id="permission-types-container">
                                <?php foreach ($permission_types as $index => $type): ?>
                                <div class="permission-type-row grid grid-cols-1 md:grid-cols-4 gap-4 p-4 border border-gray-200 rounded-lg mb-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Nama</label>
                                        <input type="text" class="type-name block w-full border border-gray-300 rounded-lg py-2 px-3" 
                                               value="<?php echo escape($type['name']); ?>" required>
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Deskripsi</label>
                                        <input type="text" class="type-description block w-full border border-gray-300 rounded-lg py-2 px-3" 
                                               value="<?php echo escape($type['description']); ?>">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-1">Maks. Durasi (Jam)</label>
                                        <input type="number" class="type-duration block w-full border border-gray-300 rounded-lg py-2 px-3" 
                                               value="<?php echo $type['max_duration_hours']; ?>" min="1" max="24">
                                    </div>
                                    <div class="flex items-end">
                                        <button type="button" onclick="removePermissionType(this)" 
                                                class="w-full bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700">
                                            <i data-feather="trash-2" class="w-4 h-4 mx-auto"></i>
                                        </button>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <div class="mt-6 flex justify-end">
                                <button onclick="savePermissionTypes()" class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700">
                                    Simpan Perubahan
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- System Tab -->
                <div id="tab-system" class="tab-content hidden">
                    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                        <!-- System Stats -->
                        <div class="bg-white rounded-lg shadow">
                            <div class="px-6 py-4 border-b border-gray-200">
                                <h2 class="text-lg font-semibold">Statistik Sistem</h2>
                            </div>
                            <div class="p-6 space-y-4">
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Ukuran Database</span>
                                    <span class="font-medium"><?php echo $stats['db_size']; ?></span>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Total Pengguna</span>
                                    <span class="font-medium"><?php echo $stats['total_users']; ?></span>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Total Izin</span>
                                    <span class="font-medium"><?php echo $stats['total_permissions']; ?></span>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Aktivitas 7 Hari</span>
                                    <span class="font-medium"><?php echo $stats['recent_activity']; ?></span>
                                </div>
                            </div>
                        </div>

                        <!-- System Actions -->
                        <div class="bg-white rounded-lg shadow">
                            <div class="px-6 py-4 border-b border-gray-200">
                                <h2 class="text-lg font-semibold">Aksi Sistem</h2>
                            </div>
                            <div class="p-6 space-y-4">
                                <form method="POST" class="space-y-4">
                                    <input type="hidden" name="action" value="backup_database">
                                    <button type="submit" class="w-full bg-blue-600 text-white px-4 py-3 rounded-lg hover:bg-blue-700 flex items-center justify-center">
                                        <i data-feather="download" class="w-5 h-5 mr-2"></i>
                                        Backup Database
                                    </button>
                                </form>
                                
                                <div class="border-t pt-4">
                                    <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                                        <div class="flex">
                                            <i data-feather="alert-triangle" class="w-5 h-5 text-yellow-600 mr-2"></i>
                                            <div>
                                                <h3 class="text-sm font-medium text-yellow-800">Peringatan</h3>
                                                <p class="text-sm text-yellow-700 mt-1">
                                                    Selalu buat backup sebelum melakukan perubahan besar pada sistem.
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="text-xs text-gray-500 space-y-1">
                                    <div>PHP Version: <?php echo phpversion(); ?></div>
                                    <div>SQLite Version: <?php echo $pdo->query('SELECT sqlite_version()')->fetchColumn(); ?></div>
                                    <div>Server: <?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
        feather.replace();
        
        function switchTab(button, tabId) {
            // Update button styles
            document.querySelectorAll('.tab-button').forEach(btn => {
                btn.classList.remove('active', 'border-blue-500', 'text-blue-600');
                btn.classList.add('border-transparent', 'text-gray-500');
            });
            
            button.classList.add('active', 'border-blue-500', 'text-blue-600');
            button.classList.remove('border-transparent', 'text-gray-500');
            
            // Update content
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.add('hidden');
            });
            
            document.getElementById('tab-' + tabId).classList.remove('hidden');
            
            // Refresh feather icons
            feather.replace();
        }
        
        function addPermissionType() {
            const container = document.getElementById('permission-types-container');
            const html = `
                <div class="permission-type-row grid grid-cols-1 md:grid-cols-4 gap-4 p-4 border border-gray-200 rounded-lg mb-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Nama</label>
                        <input type="text" class="type-name block w-full border border-gray-300 rounded-lg py-2 px-3" required>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Deskripsi</label>
                        <input type="text" class="type-description block w-full border border-gray-300 rounded-lg py-2 px-3">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Maks. Durasi (Jam)</label>
                        <input type="number" class="type-duration block w-full border border-gray-300 rounded-lg py-2 px-3" value="8" min="1" max="24">
                    </div>
                    <div class="flex items-end">
                        <button type="button" onclick="removePermissionType(this)" 
                                class="w-full bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700">
                            <i data-feather="trash-2" class="w-4 h-4 mx-auto"></i>
                        </button>
                    </div>
                </div>
            `;
            
            container.insertAdjacentHTML('beforeend', html);
            feather.replace();
        }
        
        function removePermissionType(button) {
            if (confirm('Yakin ingin menghapus jenis izin ini?')) {
                button.closest('.permission-type-row').remove();
            }
        }
        
        function savePermissionTypes() {
            const rows = document.querySelectorAll('.permission-type-row');
            const types = [];
            
            rows.forEach(row => {
                const name = row.querySelector('.type-name').value.trim();
                const description = row.querySelector('.type-description').value.trim();
                const duration = parseInt(row.querySelector('.type-duration').value);
                
                if (name) {
                    types.push({
                        name: name,
                        description: description,
                        max_duration_hours: duration
                    });
                }
            });
            
            if (types.length === 0) {
                alert('Minimal harus ada satu jenis izin!');
                return;
            }
            
            // Create form and submit
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="update_permission_types">
                <input type="hidden" name="types" value='${JSON.stringify(types)}'>
            `;
            
            document.body.appendChild(form);
            form.submit();
        }
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>