<?php
require_once 'config/config.php';

// Require login
$auth->requireLogin();

$page_title = 'Profil';
$current_user = $auth->getCurrentUser();

$error_message = '';
$success_message = '';

// Handle form submission
if ($_POST) {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error_message = 'Token keamanan tidak valid';
    } else {
        $action = $_POST['action'] ?? '';
        
        if ($action === 'update_profile') {
            // Update profile
            $name = sanitize($_POST['name'] ?? '');
            $email = sanitize($_POST['email'] ?? '');
            $phone = sanitize($_POST['phone'] ?? '');
            $position = sanitize($_POST['position'] ?? '');
            $department = sanitize($_POST['department'] ?? '');
            
            if (empty($name)) {
                $error_message = 'Nama harus diisi';
            } else {
                try {
                    $stmt = $pdo->prepare("
                        UPDATE users 
                        SET name = ?, email = ?, phone = ?, position = ?, department = ?, updated_at = CURRENT_TIMESTAMP 
                        WHERE id = ?
                    ");
                    $result = $stmt->execute([$name, $email, $phone, $position, $department, $current_user['id']]);
                    
                    if ($result) {
                        // Update session
                        $_SESSION['user_name'] = $name;
                        $_SESSION['user_email'] = $email;
                        $_SESSION['user_phone'] = $phone;
                        $_SESSION['user_position'] = $position;
                        $_SESSION['user_department'] = $department;
                        
                        $success_message = 'Profil berhasil diperbarui';
                        $current_user = $auth->getCurrentUser(); // Refresh data
                    } else {
                        $error_message = 'Gagal memperbarui profil';
                    }
                } catch(PDOException $e) {
                    $error_message = 'Terjadi kesalahan sistem: ' . $e->getMessage();
                }
            }
        } elseif ($action === 'change_password') {
            // Change password
            $old_password = $_POST['old_password'] ?? '';
            $new_password = $_POST['new_password'] ?? '';
            $confirm_password = $_POST['confirm_password'] ?? '';
            
            if (empty($old_password) || empty($new_password) || empty($confirm_password)) {
                $error_message = 'Semua field password harus diisi';
            } elseif ($new_password !== $confirm_password) {
                $error_message = 'Password baru dan konfirmasi password tidak cocok';
            } elseif (strlen($new_password) < 6) {
                $error_message = 'Password baru minimal 6 karakter';
            } else {
                $result = $auth->changePassword($current_user['id'], $old_password, $new_password);
                if ($result['success']) {
                    $success_message = $result['message'];
                } else {
                    $error_message = $result['message'];
                }
            }
        }
    }
}

$csrf_token = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <?php 
                generateBreadcrumb([
                    ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                    ['title' => 'Profil', 'url' => '']
                ]); 
                ?>
                
                <!-- Flash Messages -->
                <?php if ($error_message): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4" role="alert">
                    <div class="flex items-center">
                        <i data-feather="alert-circle" class="w-5 h-5 mr-2"></i>
                        <span><?php echo escape($error_message); ?></span>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if ($success_message): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4" role="alert">
                    <div class="flex items-center">
                        <i data-feather="check-circle" class="w-5 h-5 mr-2"></i>
                        <span><?php echo escape($success_message); ?></span>
                    </div>
                </div>
                <?php endif; ?>
                
                <div class="mb-6">
                    <h1 class="text-2xl font-bold text-gray-800">Profil Pengguna</h1>
                    <p class="text-gray-600 mt-1">Kelola informasi profil dan keamanan akun Anda</p>
                </div>

                <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    <!-- Profile Info -->
                    <div class="lg:col-span-1">
                        <div class="bg-white rounded-lg shadow p-6">
                            <div class="text-center">
                                <div class="mx-auto h-24 w-24 rounded-full bg-gray-300 flex items-center justify-center mb-4">
                                    <i data-feather="user" class="h-12 w-12 text-gray-600"></i>
                                </div>
                                <h3 class="text-lg font-medium text-gray-900"><?php echo escape($current_user['name']); ?></h3>
                                <p class="text-sm text-gray-500"><?php echo escape($current_user['position']); ?></p>
                                <p class="text-sm text-gray-500"><?php echo escape($current_user['department']); ?></p>
                                
                                <div class="mt-4 pt-4 border-t border-gray-200">
                                    <div class="flex justify-between text-sm">
                                        <span class="text-gray-500">NIP:</span>
                                        <span class="font-medium"><?php echo escape($current_user['nip']); ?></span>
                                    </div>
                                    <div class="flex justify-between text-sm mt-2">
                                        <span class="text-gray-500">Role:</span>
                                        <span class="font-medium capitalize"><?php echo escape($current_user['role']); ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Profile Form -->
                    <div class="lg:col-span-2 space-y-6">
                        <!-- Update Profile -->
                        <div class="bg-white rounded-lg shadow">
                            <div class="px-6 py-4 border-b border-gray-200">
                                <h2 class="text-lg font-semibold">Informasi Profil</h2>
                                <p class="text-sm text-gray-500">Perbarui informasi profil Anda</p>
                            </div>
                            <div class="p-6">
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                    <input type="hidden" name="action" value="update_profile">
                                    
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div>
                                            <label for="name" class="block text-sm font-medium text-gray-700 mb-1">
                                                Nama Lengkap <span class="text-red-500">*</span>
                                            </label>
                                            <input 
                                                type="text" 
                                                id="name" 
                                                name="name" 
                                                value="<?php echo escape($current_user['name']); ?>"
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                                required
                                            >
                                        </div>
                                        
                                        <div>
                                            <label for="email" class="block text-sm font-medium text-gray-700 mb-1">Email</label>
                                            <input 
                                                type="email" 
                                                id="email" 
                                                name="email" 
                                                value="<?php echo escape($current_user['email']); ?>"
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                            >
                                        </div>
                                        
                                        <div>
                                            <label for="phone" class="block text-sm font-medium text-gray-700 mb-1">Nomor Telepon</label>
                                            <input 
                                                type="tel" 
                                                id="phone" 
                                                name="phone" 
                                                value="<?php echo escape($current_user['phone']); ?>"
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                            >
                                        </div>
                                        
                                        <div>
                                            <label for="position" class="block text-sm font-medium text-gray-700 mb-1">Jabatan</label>
                                            <input 
                                                type="text" 
                                                id="position" 
                                                name="position" 
                                                value="<?php echo escape($current_user['position']); ?>"
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                            >
                                        </div>
                                        
                                        <div class="md:col-span-2">
                                            <label for="department" class="block text-sm font-medium text-gray-700 mb-1">Departemen/Bagian</label>
                                            <input 
                                                type="text" 
                                                id="department" 
                                                name="department" 
                                                value="<?php echo escape($current_user['department']); ?>"
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                            >
                                        </div>
                                    </div>
                                    
                                    <div class="mt-6 flex justify-end">
                                        <button type="submit" class="bg-blue-600 text-white px-6 py-2 rounded-lg hover:bg-blue-700 flex items-center">
                                            <i data-feather="save" class="w-4 h-4 mr-2"></i>
                                            Simpan Perubahan
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                        
                        <!-- Change Password -->
                        <div class="bg-white rounded-lg shadow">
                            <div class="px-6 py-4 border-b border-gray-200">
                                <h2 class="text-lg font-semibold">Ubah Password</h2>
                                <p class="text-sm text-gray-500">Pastikan password Anda aman dan mudah diingat</p>
                            </div>
                            <div class="p-6">
                                <form method="POST">
                                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                    <input type="hidden" name="action" value="change_password">
                                    
                                    <div class="space-y-6">
                                        <div>
                                            <label for="old_password" class="block text-sm font-medium text-gray-700 mb-1">
                                                Password Lama <span class="text-red-500">*</span>
                                            </label>
                                            <input 
                                                type="password" 
                                                id="old_password" 
                                                name="old_password" 
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                                required
                                            >
                                        </div>
                                        
                                        <div>
                                            <label for="new_password" class="block text-sm font-medium text-gray-700 mb-1">
                                                Password Baru <span class="text-red-500">*</span>
                                            </label>
                                            <input 
                                                type="password" 
                                                id="new_password" 
                                                name="new_password" 
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                                minlength="6"
                                                required
                                            >
                                            <p class="text-xs text-gray-500 mt-1">Minimal 6 karakter</p>
                                        </div>
                                        
                                        <div>
                                            <label for="confirm_password" class="block text-sm font-medium text-gray-700 mb-1">
                                                Konfirmasi Password Baru <span class="text-red-500">*</span>
                                            </label>
                                            <input 
                                                type="password" 
                                                id="confirm_password" 
                                                name="confirm_password" 
                                                class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                                required
                                            >
                                        </div>
                                    </div>
                                    
                                    <div class="mt-6 flex justify-end">
                                        <button type="submit" class="bg-red-600 text-white px-6 py-2 rounded-lg hover:bg-red-700 flex items-center">
                                            <i data-feather="key" class="w-4 h-4 mr-2"></i>
                                            Ubah Password
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
        feather.replace();
        
        // Password confirmation validation
        document.getElementById('confirm_password').addEventListener('input', function() {
            const newPassword = document.getElementById('new_password').value;
            const confirmPassword = this.value;
            
            if (newPassword && confirmPassword && newPassword !== confirmPassword) {
                this.setCustomValidity('Password tidak cocok');
            } else {
                this.setCustomValidity('');
            }
        });
        
        document.getElementById('new_password').addEventListener('input', function() {
            const confirmPassword = document.getElementById('confirm_password');
            if (confirmPassword.value) {
                confirmPassword.dispatchEvent(new Event('input'));
            }
        });
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>