<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Require login
$auth->requireLogin();

$page_title = 'Kalender Izin';
$current_user = $auth->getCurrentUser();

// Get month and year from URL parameters
$month = isset($_GET['month']) ? intval($_GET['month']) : date('n');
$year = isset($_GET['year']) ? intval($_GET['year']) : date('Y');

// Validate month and year
$month = max(1, min(12, $month));
$year = max(2020, min(2030, $year));

// Get permissions for calendar
$user_id = ($current_user['role'] === 'admin') ? null : $current_user['id'];
$calendar_permissions = $permissionManager->getPermissionsForCalendar($month, $year, $user_id);

// Group permissions by date
$permissions_by_date = [];
foreach ($calendar_permissions as $permission) {
    $date = date('j', strtotime($permission['request_date']));
    if (!isset($permissions_by_date[$date])) {
        $permissions_by_date[$date] = [];
    }
    $permissions_by_date[$date][] = $permission;
}

// Calendar helpers
function getMonthName($month) {
    $months = [
        1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April',
        5 => 'Mei', 6 => 'Juni', 7 => 'Juli', 8 => 'Agustus',
        9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
    ];
    return $months[$month];
}

function getDaysInMonth($month, $year) {
    return date('t', mktime(0, 0, 0, $month, 1, $year));
}

function getFirstDayOfWeek($month, $year) {
    return date('w', mktime(0, 0, 0, $month, 1, $year));
}

$days_in_month = getDaysInMonth($month, $year);
$first_day = getFirstDayOfWeek($month, $year);
$prev_month = $month - 1;
$prev_year = $year;
if ($prev_month < 1) {
    $prev_month = 12;
    $prev_year--;
}
$next_month = $month + 1;
$next_year = $year;
if ($next_month > 12) {
    $next_month = 1;
    $next_year++;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
        .calendar-day { min-height: 100px; }
        .permission-dot { width: 8px; height: 8px; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <?php 
                generateBreadcrumb([
                    ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                    ['title' => 'Kalender Izin', 'url' => '']
                ]); 
                ?>
                
                <!-- Flash Messages -->
                <?php showFlashMessage(); ?>
                
                <div class="mb-6">
                    <h1 class="text-2xl font-bold text-gray-800">Kalender Izin</h1>
                    <p class="text-gray-600 mt-1">Lihat jadwal izin dalam bentuk kalender</p>
                </div>

                <!-- Calendar Navigation -->
                <div class="bg-white rounded-lg shadow p-6 mb-6">
                    <div class="flex justify-between items-center mb-4">
                        <a href="?month=<?php echo $prev_month; ?>&year=<?php echo $prev_year; ?>" class="p-2 rounded-full hover:bg-gray-100">
                            <i data-feather="chevron-left" class="w-5 h-5"></i>
                        </a>
                        <h2 class="text-xl font-semibold"><?php echo getMonthName($month) . ' ' . $year; ?></h2>
                        <a href="?month=<?php echo $next_month; ?>&year=<?php echo $next_year; ?>" class="p-2 rounded-full hover:bg-gray-100">
                            <i data-feather="chevron-right" class="w-5 h-5"></i>
                        </a>
                    </div>
                    
                    <!-- Calendar Grid -->
                    <div class="grid grid-cols-7 gap-1">
                        <!-- Day headers -->
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Min</div>
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Sen</div>
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Sel</div>
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Rab</div>
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Kam</div>
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Jum</div>
                        <div class="p-3 text-center font-medium text-gray-700 bg-gray-50">Sab</div>
                        
                        <?php
                        // Empty cells for days before the first day of month
                        for ($i = 0; $i < $first_day; $i++) {
                            echo '<div class="calendar-day p-2 border border-gray-200 bg-gray-50"></div>';
                        }
                        
                        // Days of the month
                        for ($day = 1; $day <= $days_in_month; $day++) {
                            $is_today = ($day == date('j') && $month == date('n') && $year == date('Y'));
                            $has_permissions = isset($permissions_by_date[$day]);
                            
                            echo '<div class="calendar-day p-2 border border-gray-200 bg-white relative' . ($is_today ? ' bg-blue-50 border-blue-300' : '') . '">';
                            echo '<div class="text-sm font-medium' . ($is_today ? ' text-blue-600' : ' text-gray-900') . '">' . $day . '</div>';
                            
                            if ($has_permissions) {
                                foreach ($permissions_by_date[$day] as $permission) {
                                    $color_class = '';
                                    switch ($permission['permission_type_name']) {
                                        case 'Dinas Luar':
                                            $color_class = 'bg-blue-500';
                                            break;
                                        case 'Sakit':
                                            $color_class = 'bg-red-500';
                                            break;
                                        case 'Keperluan Pribadi':
                                            $color_class = 'bg-purple-500';
                                            break;
                                        case 'Cuti':
                                            $color_class = 'bg-green-500';
                                            break;
                                        default:
                                            $color_class = 'bg-gray-500';
                                    }
                                    
                                    echo '<div class="mt-1 p-1 text-xs rounded ' . $color_class . ' text-white truncate" title="' . escape($permission['user_name']) . ' - ' . escape($permission['permission_type_name']) . '">';
                                    if ($current_user['role'] === 'admin') {
                                        echo escape(substr($permission['user_name'], 0, 10));
                                    } else {
                                        echo escape(substr($permission['permission_type_name'], 0, 8));
                                    }
                                    echo '</div>';
                                }
                            }
                            echo '</div>';
                        }
                        
                        // Fill remaining cells
                        $total_cells = $first_day + $days_in_month;
                        $remaining_cells = 42 - $total_cells; // 6 weeks * 7 days
                        if ($remaining_cells > 7) $remaining_cells -= 7; // Only show 5 weeks if possible
                        
                        for ($i = 0; $i < $remaining_cells; $i++) {
                            echo '<div class="calendar-day p-2 border border-gray-200 bg-gray-50"></div>';
                        }
                        ?>
                    </div>
                </div>

                <!-- Legend & Statistics -->
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    <!-- Legend -->
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-semibold mb-4">Keterangan</h3>
                        <div class="space-y-2">
                            <div class="flex items-center space-x-2">
                                <div class="w-4 h-4 rounded bg-blue-500"></div>
                                <span class="text-sm">Dinas Luar</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <div class="w-4 h-4 rounded bg-red-500"></div>
                                <span class="text-sm">Sakit</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <div class="w-4 h-4 rounded bg-purple-500"></div>
                                <span class="text-sm">Keperluan Pribadi</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <div class="w-4 h-4 rounded bg-green-500"></div>
                                <span class="text-sm">Cuti</span>
                            </div>
                            <div class="flex items-center space-x-2">
                                <div class="w-4 h-4 rounded bg-gray-500"></div>
                                <span class="text-sm">Lainnya</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Monthly Statistics -->
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-semibold mb-4">Statistik Bulan Ini</h3>
                        <?php
                        $monthly_stats = [];
                        foreach ($calendar_permissions as $permission) {
                            $type = $permission['permission_type_name'];
                            if (!isset($monthly_stats[$type])) {
                                $monthly_stats[$type] = 0;
                            }
                            $monthly_stats[$type]++;
                        }
                        ?>
                        <div class="space-y-3">
                            <?php if (!empty($monthly_stats)): ?>
                                <?php foreach ($monthly_stats as $type => $count): ?>
                                <div class="flex justify-between items-center">
                                    <span class="text-sm text-gray-600"><?php echo escape($type); ?></span>
                                    <span class="text-sm font-medium"><?php echo $count; ?> izin</span>
                                </div>
                                <?php endforeach; ?>
                                <hr class="my-2">
                                <div class="flex justify-between items-center font-medium">
                                    <span>Total</span>
                                    <span><?php echo array_sum($monthly_stats); ?> izin</span>
                                </div>
                            <?php else: ?>
                                <p class="text-sm text-gray-500 text-center">Tidak ada izin pada bulan ini</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Quick Navigation -->
                <div class="mt-6 bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-semibold mb-4">Navigasi Cepat</h3>
                    <div class="grid grid-cols-2 md:grid-cols-6 gap-2">
                        <?php for ($m = 1; $m <= 12; $m++): ?>
                        <a href="?month=<?php echo $m; ?>&year=<?php echo $year; ?>" 
                           class="p-2 text-center rounded <?php echo ($m == $month) ? 'bg-blue-100 text-blue-600 font-medium' : 'hover:bg-gray-100'; ?>">
                            <?php echo substr(getMonthName($m), 0, 3); ?>
                        </a>
                        <?php endfor; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
        feather.replace();
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>