<?php
/**
 * Konfigurasi dan Inisialisasi Database SQLite
 * SISKIM - Sistem Izin Keluar Masuk
 * Pengadilan Negeri Watampone
 */

class Database {
    private $db_path;
    private $pdo;
    
    public function __construct() {
        $this->db_path = __DIR__ . '/../database/siskim.db';
        $this->initializeDatabase();
    }
    
    /**
     * Membuat koneksi ke database SQLite
     */
    public function getConnection() {
        try {
            if (!$this->pdo) {
                // Pastikan directory database ada
                $db_dir = dirname($this->db_path);
                if (!file_exists($db_dir)) {
                    mkdir($db_dir, 0755, true);
                }
                
                $this->pdo = new PDO('sqlite:' . $this->db_path);
                $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
                
                // Enable foreign keys
                $this->pdo->exec('PRAGMA foreign_keys = ON');
            }
            
            return $this->pdo;
        } catch(PDOException $e) {
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
    }
    
    /**
     * Inisialisasi dan pembuatan tabel database
     */
    private function initializeDatabase() {
        try {
            $pdo = $this->getConnection();
            
            // Tabel users
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS users (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    nip VARCHAR(20) UNIQUE NOT NULL,
                    name VARCHAR(100) NOT NULL,
                    email VARCHAR(100),
                    password VARCHAR(255) NOT NULL,
                    role VARCHAR(10) DEFAULT 'user' CHECK(role IN ('admin', 'user')),
                    position VARCHAR(100),
                    department VARCHAR(100),
                    phone VARCHAR(15),
                    is_active INTEGER DEFAULT 1,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Tabel permission_types
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS permission_types (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    name VARCHAR(50) NOT NULL,
                    description TEXT,
                    is_active INTEGER DEFAULT 1,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Tabel permissions (pengajuan izin)
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS permissions (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    user_id INTEGER NOT NULL,
                    permission_type_id INTEGER NOT NULL,
                    request_date DATE NOT NULL,
                    start_time TIME NOT NULL,
                    end_time TIME NOT NULL,
                    purpose TEXT NOT NULL,
                    destination TEXT NOT NULL,
                    document_path VARCHAR(255),
                    status VARCHAR(10) DEFAULT 'pending' CHECK(status IN ('pending', 'approved', 'rejected')),
                    approved_by INTEGER,
                    approved_at DATETIME,
                    rejection_reason TEXT,
                    notes TEXT,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                    FOREIGN KEY (permission_type_id) REFERENCES permission_types(id),
                    FOREIGN KEY (approved_by) REFERENCES users(id)
                )
            ");
            
            // Tabel settings
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS settings (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    setting_key VARCHAR(100) UNIQUE NOT NULL,
                    setting_value TEXT,
                    description TEXT,
                    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
                )
            ");
            
            // Insert data default jika belum ada
            $this->insertDefaultData();
            
        } catch(PDOException $e) {
            throw new Exception("Database initialization failed: " . $e->getMessage());
        }
    }
    
    /**
     * Insert data default untuk sistem
     */
    private function insertDefaultData() {
        try {
            $pdo = $this->getConnection();
            
            // Check apakah sudah ada data admin
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE role = 'admin'");
            $stmt->execute();
            $adminCount = $stmt->fetchColumn();
            
            if ($adminCount == 0) {
                // Insert admin default
                $stmt = $pdo->prepare("
                    INSERT INTO users (nip, name, email, password, role, position, department) 
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    'admin',
                    'Administrator',
                    'admin@pn-watampone.go.id',
                    password_hash('admin123', PASSWORD_DEFAULT),
                    'admin',
                    'Administrator Sistem',
                    'IT'
                ]);
                
                // Insert user demo
                $stmt->execute([
                    '199606192025062007',
                    'John Doe',
                    'john.doe@pn-watampone.go.id',
                    password_hash('123456', PASSWORD_DEFAULT),
                    'user',
                    'Hakim',
                    'Perdata'
                ]);
            }
            
            // Check dan insert permission types
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM permission_types");
            $stmt->execute();
            $typeCount = $stmt->fetchColumn();
            
            if ($typeCount == 0) {
                $permission_types = [
                    ['Dinas Luar', 'Izin keluar untuk keperluan dinas resmi'],
                    ['Keperluan Pribadi', 'Izin keluar untuk urusan pribadi'],
                    ['Sakit', 'Izin tidak masuk karena sakit'],
                    ['Cuti', 'Izin cuti tahunan atau khusus'],
                    ['Lainnya', 'Jenis izin lainnya']
                ];
                
                $stmt = $pdo->prepare("INSERT INTO permission_types (name, description) VALUES (?, ?)");
                foreach ($permission_types as $type) {
                    $stmt->execute($type);
                }
            }
            
            // Check dan insert settings default
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM settings");
            $stmt->execute();
            $settingCount = $stmt->fetchColumn();
            
            if ($settingCount == 0) {
                $settings = [
                    ['app_name', 'SISKIM - Pengadilan Negeri Watampone', 'Nama aplikasi'],
                    ['app_version', '1.0.0', 'Versi aplikasi'],
                    ['max_file_size', '5242880', 'Ukuran maksimal file upload (5MB)'],
                    ['allowed_file_types', 'pdf,jpg,jpeg,png', 'Tipe file yang diizinkan'],
                    ['work_start_time', '08:00', 'Jam masuk kerja'],
                    ['work_end_time', '16:00', 'Jam pulang kerja'],
                    ['auto_approve_duration', '2', 'Durasi izin yang bisa di-approve otomatis (jam)']
                ];
                
                $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value, description) VALUES (?, ?, ?)");
                foreach ($settings as $setting) {
                    $stmt->execute($setting);
                }
            }
            
        } catch(PDOException $e) {
            // Jika ada error saat insert default data, lanjutkan saja
            error_log("Warning: " . $e->getMessage());
        }
    }
    
    /**
     * Get setting value by key
     */
    public function getSetting($key) {
        try {
            $pdo = $this->getConnection();
            $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
            $stmt->execute([$key]);
            return $stmt->fetchColumn();
        } catch(PDOException $e) {
            return null;
        }
    }
    
    /**
     * Set setting value
     */
    public function setSetting($key, $value) {
        try {
            $pdo = $this->getConnection();
            $stmt = $pdo->prepare("
                INSERT OR REPLACE INTO settings (setting_key, setting_value, updated_at) 
                VALUES (?, ?, CURRENT_TIMESTAMP)
            ");
            return $stmt->execute([$key, $value]);
        } catch(PDOException $e) {
            return false;
        }
    }
}

// Instance global database
$database = new Database();
$pdo = $database->getConnection();
?>