<?php
/**
 * Konfigurasi Utama Aplikasi
 * SISKIM - Sistem Izin Keluar Masuk
 * Pengadilan Negeri Watampone
 */

// Error reporting (untuk development, ubah ke 0 untuk production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Timezone
date_default_timezone_set('Asia/Makassar');

// Session configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 0); // Set ke 1 jika menggunakan HTTPS
ini_set('session.use_strict_mode', 1);

// Start session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Include database and auth
require_once __DIR__ . '/database.php';
require_once __DIR__ . '/auth.php';

// App constants
define('APP_NAME', 'SISKIM');
define('APP_VERSION', '1.0.0');
define('APP_URL', 'http://localhost/ijinkeluar');
define('UPLOAD_PATH', __DIR__ . '/../uploads/');
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
define('ALLOWED_FILE_TYPES', ['pdf', 'jpg', 'jpeg', 'png']);

// Create upload directory if not exists
if (!file_exists(UPLOAD_PATH)) {
    mkdir(UPLOAD_PATH, 0755, true);
}

/**
 * Helper Functions
 */

// Fungsi untuk format tanggal Indonesia
function formatTanggalIndonesia($date, $format = 'd F Y') {
    $months = [
        1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April',
        5 => 'Mei', 6 => 'Juni', 7 => 'Juli', 8 => 'Agustus',
        9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
    ];
    
    $timestamp = strtotime($date);
    $day = date('d', $timestamp);
    $month = $months[(int)date('m', $timestamp)];
    $year = date('Y', $timestamp);
    
    if ($format === 'd F Y') {
        return "$day $month $year";
    } elseif ($format === 'd F Y H:i') {
        $time = date('H:i', $timestamp);
        return "$day $month $year $time";
    }
    
    return date($format, $timestamp);
}

// Fungsi untuk format waktu
function formatWaktu($time) {
    return date('H:i', strtotime($time));
}

// Fungsi untuk menghitung durasi
function hitungDurasi($start_time, $end_time) {
    $start = strtotime($start_time);
    $end = strtotime($end_time);
    $diff = $end - $start;
    
    $hours = floor($diff / 3600);
    $minutes = floor(($diff % 3600) / 60);
    
    if ($hours > 0) {
        return $hours . ' jam' . ($minutes > 0 ? ' ' . $minutes . ' menit' : '');
    } else {
        return $minutes . ' menit';
    }
}

// Fungsi untuk upload file
function uploadFile($file, $allowed_types = ALLOWED_FILE_TYPES, $max_size = MAX_FILE_SIZE) {
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'message' => 'Error uploading file'];
    }
    
    // Check file size
    if ($file['size'] > $max_size) {
        return ['success' => false, 'message' => 'File terlalu besar. Maksimal ' . ($max_size / 1024 / 1024) . 'MB'];
    }
    
    // Check file type
    $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($file_extension, $allowed_types)) {
        return ['success' => false, 'message' => 'Tipe file tidak diizinkan. Hanya: ' . implode(', ', $allowed_types)];
    }
    
    // Generate unique filename
    $filename = uniqid() . '.' . $file_extension;
    $filepath = UPLOAD_PATH . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return [
            'success' => true, 
            'filename' => $filename,
            'filepath' => $filepath
        ];
    } else {
        return ['success' => false, 'message' => 'Gagal menyimpan file'];
    }
}

// Fungsi untuk delete file
function deleteFile($filename) {
    $filepath = UPLOAD_PATH . $filename;
    if (file_exists($filepath)) {
        return unlink($filepath);
    }
    return true;
}

// Fungsi untuk redirect dengan message
function redirect($url, $message = '', $type = 'success') {
    if (!empty($message)) {
        $_SESSION['flash_message'] = $message;
        $_SESSION['flash_type'] = $type;
    }
    header("Location: $url");
    exit();
}

// Fungsi untuk show flash message
function showFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'success';
        
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);
        
        $bg_color = $type === 'success' ? 'bg-green-100 text-green-700 border-green-300' : 'bg-red-100 text-red-700 border-red-300';
        $icon = $type === 'success' ? 'check-circle' : 'x-circle';
        
        echo "<div class='flash-message $bg_color border px-4 py-3 rounded mb-4' role='alert'>";
        echo "<div class='flex items-center'>";
        echo "<i data-feather='$icon' class='w-5 h-5 mr-2'></i>";
        echo "<span>$message</span>";
        echo "<button onclick='this.parentElement.parentElement.remove()' class='ml-auto'>";
        echo "<i data-feather='x' class='w-4 h-4'></i>";
        echo "</button>";
        echo "</div>";
        echo "</div>";
    }
}

// Fungsi untuk generate breadcrumb
function generateBreadcrumb($items) {
    echo '<nav class="flex mb-4" aria-label="Breadcrumb">';
    echo '<ol class="inline-flex items-center space-x-1 md:space-x-3">';
    
    foreach ($items as $index => $item) {
        $isLast = $index === count($items) - 1;
        
        echo '<li class="inline-flex items-center">';
        
        if ($index > 0) {
            echo '<i data-feather="chevron-right" class="w-4 h-4 text-gray-400 mx-1"></i>';
        }
        
        if ($isLast) {
            echo '<span class="text-sm font-medium text-gray-500">' . escape($item['title']) . '</span>';
        } else {
            echo '<a href="' . escape($item['url']) . '" class="inline-flex items-center text-sm font-medium text-gray-700 hover:text-blue-600">';
            if (isset($item['icon'])) {
                echo '<i data-feather="' . $item['icon'] . '" class="w-4 h-4 mr-1"></i>';
            }
            echo escape($item['title']);
            echo '</a>';
        }
        
        echo '</li>';
    }
    
    echo '</ol>';
    echo '</nav>';
}

// Fungsi untuk pagination
function generatePagination($current_page, $total_pages, $url_base) {
    if ($total_pages <= 1) return '';
    
    $pagination = '<div class="flex justify-center mt-6">';
    $pagination .= '<nav class="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">';
    
    // Previous button
    $prev_disabled = $current_page <= 1 ? 'pointer-events-none opacity-50' : '';
    $prev_page = max(1, $current_page - 1);
    $pagination .= "<a href='$url_base?page=$prev_page' class='$prev_disabled relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50'>";
    $pagination .= '<i data-feather="chevron-left" class="w-4 h-4"></i>';
    $pagination .= '</a>';
    
    // Page numbers
    $start = max(1, $current_page - 2);
    $end = min($total_pages, $current_page + 2);
    
    for ($i = $start; $i <= $end; $i++) {
        $active = $i === $current_page ? 'bg-blue-50 border-blue-500 text-blue-600' : 'border-gray-300 text-gray-500 hover:bg-gray-50';
        $pagination .= "<a href='$url_base?page=$i' class='$active relative inline-flex items-center px-4 py-2 border text-sm font-medium'>$i</a>";
    }
    
    // Next button
    $next_disabled = $current_page >= $total_pages ? 'pointer-events-none opacity-50' : '';
    $next_page = min($total_pages, $current_page + 1);
    $pagination .= "<a href='$url_base?page=$next_page' class='$next_disabled relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50'>";
    $pagination .= '<i data-feather="chevron-right" class="w-4 h-4"></i>';
    $pagination .= '</a>';
    
    $pagination .= '</nav>';
    $pagination .= '</div>';
    
    return $pagination;
}

/**
 * Status constants
 */
define('STATUS_PENDING', 'pending');
define('STATUS_APPROVED', 'approved');
define('STATUS_REJECTED', 'rejected');

function getStatusBadge($status) {
    switch ($status) {
        case STATUS_APPROVED:
            return '<span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-600">Disetujui</span>';
        case STATUS_PENDING:
            return '<span class="px-2 py-1 text-xs rounded-full bg-yellow-100 text-yellow-600">Menunggu</span>';
        case STATUS_REJECTED:
            return '<span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-600">Ditolak</span>';
        default:
            return '<span class="px-2 py-1 text-xs rounded-full bg-gray-100 text-gray-600">Unknown</span>';
    }
}

// Image management functions
function getImagePath($filename) {
    return 'img.php?img=' . urlencode($filename);
}

function getImageUrl($filename) {
    return APP_URL . '/img.php?img=' . urlencode($filename);
}

function displayImage($filename, $alt = '', $class = '') {
    if (file_exists(__DIR__ . '/../gambar/' . $filename)) {
        return '<img src="' . getImagePath($filename) . '" alt="' . escape($alt) . '" class="' . $class . '">';
    }
    return '<div class="' . $class . ' bg-gray-200 flex items-center justify-center text-gray-500 text-sm">Gambar tidak tersedia</div>';
}

// Logo functions
function getLogoHtml($class = 'h-8') {
    return displayImage('Logo_pnwtp.png', 'Logo PN Watampone', $class);
}

function getKopSuratHtml($class = 'w-full') {
    return displayImage('kop_surat.png', 'Kop Surat', $class);
}

function getGedungHtml($class = 'w-full') {
    return displayImage('gedung-pn-wtp.png', 'Gedung PN Watampone', $class);
}
?>